// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#pragma once

#include "CoreMinimal.h"
#include "Dom/JsonObject.h"

#include "MultiverseSDKClasses.generated.h"

USTRUCT(BlueprintType)
struct FMultiverseObjectMeta
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FString Name;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FString Namespace;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FString Uid;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FString ResourceVersion;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	int64 Generation = 0;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	int64 CreationTimestamp = 0;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	int64 DeletionTimestamp = 0;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	TMap<FString, FString> Annotations;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	TMap<FString, FString> Labels;

	FMultiverseObjectMeta()
	{
	}

	explicit FMultiverseObjectMeta(TSharedPtr<FJsonObject> JsonObject)
	{
		JsonObject->TryGetStringField(TEXT("name"), Name);
		JsonObject->TryGetStringField(TEXT("namespace"), Namespace);
		JsonObject->TryGetStringField(TEXT("uid"), Uid);
		JsonObject->TryGetStringField(TEXT("resource_version"), ResourceVersion);
		JsonObject->TryGetNumberField(TEXT("generation"), Generation);
		JsonObject->TryGetNumberField(TEXT("creation_timestamp"), CreationTimestamp);
		JsonObject->TryGetNumberField(TEXT("deletion_timestamp"), DeletionTimestamp);
		const TSharedPtr<FJsonObject>* AnnotationsJsonObject;
		if (JsonObject->TryGetObjectField(TEXT("annotations"), AnnotationsJsonObject))
		{
			for (const auto& Entry : (*AnnotationsJsonObject)->Values)
			{
				if (Entry.Value.IsValid() && !Entry.Value->IsNull())
				{
					FJsonValueString Key = Entry.Key;
					TSharedPtr<FJsonValue> Value = Entry.Value;
					FString AnnotationKey = Key.AsString();
					FString AnnotationValue = Value->AsString();
					Annotations.Add(AnnotationKey, AnnotationValue);
				}
			}
		}
		const TSharedPtr<FJsonObject>* LabelsObject;
		if (JsonObject->TryGetObjectField(TEXT("labels"), LabelsObject))
		{
			for (const auto& Entry : (*LabelsObject)->Values)
			{
				if (Entry.Value.IsValid() && !Entry.Value->IsNull())
				{
					FJsonValueString Key = Entry.Key;
					TSharedPtr<FJsonValue> Value = Entry.Value;
					FString LabelKey = Key.AsString();
					FString LabelValue = Value->AsString();
					Labels.Add(LabelKey, LabelValue);
				}
			}
		}
	}
};

USTRUCT(BlueprintType)
struct FMultiverseHealth
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	bool bDisabled = false;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	int32 PeriodSeconds = 0;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	int32 FailureThreshold = 0;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	int32 InitialDelaySeconds = 0;

	FMultiverseHealth()
	{
	}

	explicit FMultiverseHealth(const TSharedPtr<FJsonObject> JsonObject)
	{
		JsonObject->TryGetBoolField(TEXT("disabled"), bDisabled);
		JsonObject->TryGetNumberField(TEXT("period_seconds"), PeriodSeconds);
		JsonObject->TryGetNumberField(TEXT("failure_threshold"), FailureThreshold);
		JsonObject->TryGetNumberField(TEXT("initial_delay_seconds"), InitialDelaySeconds);
	}
};

USTRUCT(BlueprintType)
struct FMultiverseSpec
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FMultiverseHealth Health;

	UPROPERTY(BlueprintReadOnly, Category = "MultiverseSDK")
	TMap<FString, FString> Env;

	FMultiverseSpec()
	{
	}

	explicit FMultiverseSpec(const TSharedPtr<FJsonObject> JsonObject)
	{
		const TSharedPtr<FJsonObject>* HealthJsonObject;
		if (JsonObject->TryGetObjectField(TEXT("health"), HealthJsonObject))
		{
			Health = FMultiverseHealth(*HealthJsonObject);
		}
		const TSharedPtr<FJsonObject>* EnvsObject;
		if (JsonObject->TryGetObjectField(TEXT("env"), EnvsObject))
		{
			for (const auto& Entry : (*EnvsObject)->Values)
			{
				if (Entry.Value.IsValid() && !Entry.Value->IsNull())
				{
					FJsonValueString Key = Entry.Key;
					TSharedPtr<FJsonValue> Value = Entry.Value;
					FString EnvKey = Key.AsString();
					FString EnvValue = Value->AsString();
					Env.Add(EnvKey, EnvValue);
				}
			}
		}
	}
};

USTRUCT(BlueprintType)
struct FMultiversePort
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FString Name;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	int32 Port = 0;

	FMultiversePort()
	{
	}

	explicit FMultiversePort(const TSharedPtr<FJsonObject> JsonObject)
	{
		JsonObject->TryGetStringField(TEXT("name"), Name);
		JsonObject->TryGetNumberField(TEXT("port"), Port);
	}
};

USTRUCT(BlueprintType)
struct FMultiverseStatus
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FString State;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FString Address;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	TArray<FMultiversePort> Ports;

	FMultiverseStatus()
	{
	}

	explicit FMultiverseStatus(const TSharedPtr<FJsonObject> JsonObject)
	{
		JsonObject->TryGetStringField(TEXT("state"), State);
		JsonObject->TryGetStringField(TEXT("address"), Address);
		const TArray<TSharedPtr<FJsonValue>>* PortsArray;
		if (JsonObject->TryGetArrayField(TEXT("ports"), PortsArray))
		{
			const int32 ArrLen = PortsArray->Num();
			for (int32 i = 0; i < ArrLen; ++i)
			{
				const TSharedPtr<FJsonValue>& PortItem = (*PortsArray)[i];
				if (PortItem.IsValid() && !PortItem->IsNull())
				{
					FMultiversePort Port = FMultiversePort(PortItem->AsObject());
					Ports.Add(Port);
				}
			}
		}
	}
};

USTRUCT(BlueprintType)
struct FGetLabelsResponse
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category = "MultiverseSDK");
	TMap<FString, FString> Labels;

	FGetLabelsResponse() {}

	explicit FGetLabelsResponse(const TSharedPtr<FJsonObject> JsonObject)
	{
		const TSharedPtr<FJsonObject>* ObjectMetaJsonObject;
		Labels = TMap<FString, FString>();
		if (JsonObject->TryGetObjectField(TEXT("object_meta"), ObjectMetaJsonObject))
		{
			struct FMultiverseObjectMeta ObjectMeta = FMultiverseObjectMeta(*ObjectMetaJsonObject);
			for (auto x : ObjectMeta.Labels) {
				Labels.Add(x.Key, x.Value);
			}
		}
	}

};

USTRUCT(BlueprintType)
struct FGetLabelEnvsResponse
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category = "MultiverseSDK");
	TMap<FString, FString> LabelEnvs;

	FGetLabelEnvsResponse() {}

	explicit FGetLabelEnvsResponse(const TSharedPtr<FJsonObject> JsonObject)
	{
		LabelEnvs = TMap<FString, FString>();
		const TSharedPtr<FJsonObject>* ObjectMetaJsonObject;
		if (JsonObject->TryGetObjectField(TEXT("object_meta"), ObjectMetaJsonObject))
		{
			struct FMultiverseObjectMeta ObjectMeta = FMultiverseObjectMeta(*ObjectMetaJsonObject);
			for (auto x : ObjectMeta.Labels) {
				LabelEnvs.Add(x.Key, x.Value);
			}
		}
		const TSharedPtr<FJsonObject>* SpecJsonObject;
		if (JsonObject->TryGetObjectField(TEXT("spec"), SpecJsonObject))
		{
			struct FMultiverseSpec Spec = FMultiverseSpec(*SpecJsonObject);
			for (auto x : Spec.Env) {
				LabelEnvs.Add(x.Key, x.Value);
			}
		}
		
	}

};

USTRUCT(BlueprintType)
struct FMultiverseGameServerResponse
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FMultiverseStatus Status;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FMultiverseObjectMeta ObjectMeta;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FMultiverseSpec Spec;

	FMultiverseGameServerResponse()
	{
	}

	explicit FMultiverseGameServerResponse(const TSharedPtr<FJsonObject> JsonObject)
	{
		const TSharedPtr<FJsonObject>* ObjectMetaJsonObject;
		if (JsonObject->TryGetObjectField(TEXT("object_meta"), ObjectMetaJsonObject))
		{
			ObjectMeta = FMultiverseObjectMeta(*ObjectMetaJsonObject);
		}
		const TSharedPtr<FJsonObject>* SpecJsonObject;
		if (JsonObject->TryGetObjectField(TEXT("spec"), SpecJsonObject))
		{
			Spec = FMultiverseSpec(*SpecJsonObject);
		}
		const TSharedPtr<FJsonObject>* StatusJsonObject;
		if (JsonObject->TryGetObjectField(TEXT("status"), StatusJsonObject))
		{
			Status = FMultiverseStatus(*StatusJsonObject);
		}
	}
};

USTRUCT(BlueprintType)
struct FMultiverseKeyValuePair
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FString Key;

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FString Value;
};

USTRUCT(BlueprintType)
struct FMultiverseDuration
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	int64 Seconds;
};

USTRUCT(BlueprintType)
struct FMultiverseSDKPlayer
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FString PlayerID;
};

USTRUCT(BlueprintType)
struct FMultiversePlayerCapacity
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	int64 Count;
};

USTRUCT(BlueprintType)
struct FMultiverseEmptyResponse
{
	GENERATED_BODY()
};

USTRUCT(BlueprintType)
struct FMultiverseSDKError
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	FString ErrorMessage;
};

USTRUCT(BlueprintType)
struct FMultiverseConnectedResponse
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	bool bConnected = false;

	FMultiverseConnectedResponse()
	{
	}

	explicit FMultiverseConnectedResponse(const TSharedPtr<FJsonObject> JsonObject)
	{
		JsonObject->TryGetBoolField(TEXT("bool"), bConnected);
	}
};

USTRUCT(BlueprintType)
struct FMultiverseDisconnectResponse
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	bool bDisconnected = false;

	FMultiverseDisconnectResponse()
	{
	}

	explicit FMultiverseDisconnectResponse(const TSharedPtr<FJsonObject> JsonObject)
	{
		JsonObject->TryGetBoolField(TEXT("bool"), bDisconnected);
	}
};

USTRUCT(BlueprintType)
struct FMultiverseCountResponse
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	int64 Count = 0;

	FMultiverseCountResponse()
	{
	}

	explicit FMultiverseCountResponse(const TSharedPtr<FJsonObject> JsonObject)
	{
		JsonObject->TryGetNumberField(TEXT("count"), Count);
	}
};

USTRUCT(BlueprintType)
struct FMultiverseConnectedPlayersResponse
{
	GENERATED_BODY()

	FMultiverseConnectedPlayersResponse()
	{
	}

	UPROPERTY(BlueprintReadOnly, Category="MultiverseSDK")
	TArray<FString> ConnectedPlayers;

	explicit FMultiverseConnectedPlayersResponse(const TSharedPtr<FJsonObject> JsonObject)
	{
		JsonObject->TryGetStringArrayField(TEXT("list"), ConnectedPlayers);
	}
};
