// Fill out your copyright notice in the Description page of Project Settings.


#include "MultiverseSDKComponent.h"

#include "Engine/World.h"
#include "HttpModule.h"
#include "Interfaces/IHttpResponse.h"
#include "JsonUtilities/Public/JsonObjectConverter.h"
#include "TimerManager.h"
#include "WebSockets/Public/IWebSocket.h"
#include "WebSockets/Public/WebSocketsModule.h"
// Sets default values for this component's properties
UMultiverseSDKComponent::UMultiverseSDKComponent()
{
	// Set this component to be initialized when the game starts, and to be ticked every frame.  You can turn these features
	// off to improve performance if you don't need them.
	PrimaryComponentTick.bCanEverTick = false;

	// ...
}


// Called when the game starts
void UMultiverseSDKComponent::BeginPlay()
{
	Super::BeginPlay();

	// ...
	HealthPing(HealthRateSeconds);
	if (bDisableAutoConnect)
	{
		return;
	}
	Connect();
}


void UMultiverseSDKComponent::EndPlay(const EEndPlayReason::Type EndPlayReason)
{
	Super::EndPlay(EndPlayReason);

	const UWorld* World = GetWorld();
	if (World != nullptr)
	{
		World->GetTimerManager().ClearTimer(ConnectDelTimerHandle);
		World->GetTimerManager().ClearTimer(HealthTimerHandler);
		World->GetTimerManager().ClearTimer(EnsureWebSocketTimerHandler);
	}

	if (WatchWebSocket != nullptr && WatchWebSocket->IsConnected())
	{
		WatchWebSocket->Close();
	}
}

FHttpRequestRef UMultiverseSDKComponent::BuildMultiverseRequest(const FString Path, const FMultiverseHttpVerb Verb, const FString Content)
{
	FHttpModule* Http = &FHttpModule::Get();
	FHttpRequestRef Request = Http->CreateRequest();

	Request->SetURL(FString::Format(
		TEXT("http://localhost:{0}/{1}"),
		{ FStringFormatArg(HttpPort), FStringFormatArg(Path) }
	));
	Request->SetVerb(Verb.ToString());
	Request->SetHeader(TEXT("Content-Type"), TEXT("application/json"));
	Request->SetHeader(TEXT("User-Agent"), TEXT("X-UnrealEngine-Agent"));
	Request->SetHeader(TEXT("Accepts"), TEXT("application/json"));
	Request->SetContentAsString(Content);
	return Request;
}

void UMultiverseSDKComponent::HealthPing(const float RateSeconds)
{
	if (RateSeconds <= 0.0f)
	{
		return;
	}

	FTimerDelegate TimerDel;
	TimerDel.BindUObject(this, &UMultiverseSDKComponent::Health, FHealthDelegate(), FMultiverseErrorDelegate());
	GetWorld()->GetTimerManager().ClearTimer(HealthTimerHandler);
	GetWorld()->GetTimerManager().SetTimer(HealthTimerHandler, TimerDel, RateSeconds, true);
}

// Called every frame
void UMultiverseSDKComponent::TickComponent(float DeltaTime, ELevelTick TickType, FActorComponentTickFunction* ThisTickFunction)
{
	Super::TickComponent(DeltaTime, TickType, ThisTickFunction);

	// ...
}

void UMultiverseSDKComponent::Connect()
{
	FGameServerDelegate SuccessDel;
	SuccessDel.BindUFunction(this, FName("ConnectSuccess"));
	FTimerDelegate ConnectDel;
	ConnectDel.BindUObject(this, &UMultiverseSDKComponent::GameServer, SuccessDel, FMultiverseErrorDelegate());
	GetWorld()->GetTimerManager().ClearTimer(ConnectDelTimerHandle);
	GetWorld()->GetTimerManager().SetTimer(ConnectDelTimerHandle, ConnectDel, 5.f, true);
}

void UMultiverseSDKComponent::ConnectSuccess(const FMultiverseGameServerResponse GameServerResponse)
{
	GetWorld()->GetTimerManager().ClearTimer(ConnectDelTimerHandle);
	Ready({}, {});
	ConnectedDelegate.Broadcast(GameServerResponse);
}

void UMultiverseSDKComponent::Ready(const FReadyDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	FHttpRequestRef Request = BuildMultiverseRequest("ready");
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			if (!IsValidResponse(bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound({});
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::GameServer(const FGameServerDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	FHttpRequestRef Request = BuildMultiverseRequest("gameserver", FMultiverseHttpVerb::Get, "");
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, const FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			TSharedPtr<FJsonObject> JsonObject;

			if (!IsValidJsonResponse(JsonObject, bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound(FMultiverseGameServerResponse(JsonObject));
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::GetLabels(const FGetLabelsDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	FHttpRequestRef Request = BuildMultiverseRequest("gameserver", FMultiverseHttpVerb::Get, "");
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, const FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			TSharedPtr<FJsonObject> JsonObject;

			if (!IsValidJsonResponse(JsonObject, bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound(FGetLabelsResponse(JsonObject));
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::GetLabelEnvs(const FGetLabelEnvsDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	FHttpRequestRef Request = BuildMultiverseRequest("gameserver", FMultiverseHttpVerb::Get, "");
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, const FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			TSharedPtr<FJsonObject> JsonObject;

			if (!IsValidJsonResponse(JsonObject, bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound(FGetLabelEnvsResponse(JsonObject));
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::EnsureWebSocketConnection()
{
	if (WatchWebSocket == nullptr)
	{
		if (!FModuleManager::LoadModulePtr<FWebSocketsModule>(TEXT("WebSockets")))
		{
			return;
		}

		TMap<FString, FString> Headers;

		// Make up a WebSocket-Key value. It can be anything!
		Headers.Add(TEXT("Sec-WebSocket-Key"), FGuid::NewGuid().ToString(EGuidFormats::Short));
		Headers.Add(TEXT("Sec-WebSocket-Version"), TEXT("13"));
		Headers.Add(TEXT("User-Agent"), TEXT("X-UnrealEngine-Agent"));

		// Unreal WebSockets are not able to do DNS resolution for localhost for some reason
		// so this is using the IPv4 Loopback Address instead.
		WatchWebSocket = FWebSocketsModule::Get().CreateWebSocket(
			FString::Format(TEXT("ws://127.0.0.1:{0}/watch/gameserver"),
				static_cast<FStringFormatOrderedArguments>(
					TArray<FStringFormatArg, TFixedAllocator<1>>{
			FStringFormatArg(HttpPort)
		}
		)
			),
			TEXT("")
			);

		WatchWebSocket->OnRawMessage().AddUObject(this, &UMultiverseSDKComponent::HandleWatchMessage);
	}

	if (WatchWebSocket != nullptr)
	{
		if (!WatchWebSocket->IsConnected())
		{
			WatchWebSocket->Connect();
		}

		// Only start the timer if there is a websocket to check.
		// This timer has nothing to do with health and only matters if the agent is somehow
		// restarted, which would be a failure condition in normal operation.
		if (!EnsureWebSocketTimerHandler.IsValid())
		{
			FTimerDelegate TimerDel;
			TimerDel.BindUObject(this, &UMultiverseSDKComponent::EnsureWebSocketConnection);
			GetWorld()->GetTimerManager().SetTimer(
				EnsureWebSocketTimerHandler, TimerDel, 15.0f, true);
		}
	}
}

void UMultiverseSDKComponent::WatchGameServer(const FGameServerDelegate WatchDelegate)
{
	WatchGameServerCallbacks.Add(WatchDelegate);
	EnsureWebSocketConnection();
}

void UMultiverseSDKComponent::DeserializeAndBroadcastWatch(FString const& JsonString)
{
	TSharedRef<TJsonReader<TCHAR>> const JsonReader = TJsonReaderFactory<TCHAR>::Create(JsonString);

	TSharedPtr<FJsonObject> JsonObject;
	const TSharedPtr<FJsonObject>* ResultObject = nullptr;

	if (!FJsonSerializer::Deserialize(JsonReader, JsonObject) ||
		!JsonObject.IsValid() ||
		!JsonObject->TryGetObjectField(TEXT("result"), ResultObject) ||
		!ResultObject->IsValid())
	{
		return;
	}

	FMultiverseGameServerResponse const Result = FMultiverseGameServerResponse(*ResultObject);
	for (FGameServerDelegate const& Callback : WatchGameServerCallbacks)
	{
		if (Callback.IsBound())
		{
			Callback.Execute(Result);
		}
	}
}

void UMultiverseSDKComponent::HandleWatchMessage(const void* Data, SIZE_T Size, SIZE_T BytesRemaining)
{
	if (BytesRemaining > 0)
	{
		WatchMessageBuffer.Append(UTF8_TO_TCHAR(static_cast<const UTF8CHAR*>(Data)), Size);
		return;
	}

	FString const Message = FString(Size, UTF8_TO_TCHAR(static_cast<const UTF8CHAR*>(Data)));

	// If the LHS of FString + is empty, it just uses the RHS directly so there's no copy here with an empty buffer.
	DeserializeAndBroadcastWatch(WatchMessageBuffer + Message);

	// Faster to check and then empty vs blindly emptying - normal case is that the buffer is already empty
	if (!WatchMessageBuffer.IsEmpty())
	{
		WatchMessageBuffer.Empty();
	}
}

void UMultiverseSDKComponent::SetLabel(
	const FString& Key, const FString& Value, const FSetLabelDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	const FMultiverseKeyValuePair Label = { Key, Value };
	FString Json;
	if (!FJsonObjectConverter::UStructToJsonObjectString(Label, Json))
	{
		ErrorDelegate.ExecuteIfBound({ FString::Format(TEXT("error serializing key-value pair ({0}: {1}})"),
			static_cast<FStringFormatOrderedArguments>(
			TArray<FStringFormatArg, TFixedAllocator<2>>{
				FStringFormatArg(Key),
				FStringFormatArg(Value)
			})
		) });
		return;
	}

	FHttpRequestRef Request = BuildMultiverseRequest("metadata/label", FMultiverseHttpVerb::Put, Json);
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, const FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			if (!IsValidResponse(bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound({});
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::Health(const FHealthDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	FHttpRequestRef Request = BuildMultiverseRequest("health");
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, const FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			if (!IsValidResponse(bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound({});
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::Shutdown(const FShutdownDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	FHttpRequestRef Request = BuildMultiverseRequest("shutdown");
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			if (!IsValidResponse(bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound({});
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::SetAnnotation(
	const FString& Key, const FString& Value, const FSetAnnotationDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	const FMultiverseKeyValuePair Label = { Key, Value };
	FString Json;
	if (!FJsonObjectConverter::UStructToJsonObjectString(Label, Json))
	{
		ErrorDelegate.ExecuteIfBound({ FString::Format(TEXT("error serializing key-value pair ({0}: {1}})"),
			static_cast<FStringFormatOrderedArguments>(
				TArray<FStringFormatArg, TFixedAllocator<2>>{
					FStringFormatArg(Key),
					FStringFormatArg(Value)
				}
			)
		) });
		return;
	}

	FHttpRequestRef Request = BuildMultiverseRequest("metadata/annotation", FMultiverseHttpVerb::Put, Json);
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			if (!IsValidResponse(bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound({});
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::Allocate(const FAllocateDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	FHttpRequestRef Request = BuildMultiverseRequest("allocate");
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			if (!IsValidResponse(bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound({});
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::Reserve(
	const int64 Seconds, const FReserveDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	const FMultiverseDuration Duration = { Seconds };
	FString Json;
	if (!FJsonObjectConverter::UStructToJsonObjectString(Duration, Json))
	{
		ErrorDelegate.ExecuteIfBound({ TEXT("Failed to serializing request") });
		return;
	}

	FHttpRequestRef Request = BuildMultiverseRequest("reserve", FMultiverseHttpVerb::Post, Json);
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			if (!IsValidResponse(bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound({});
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::PlayerConnect(
	const FString PlayerId, const FPlayerConnectDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	const FMultiverseSDKPlayer Player = { PlayerId };
	FString Json;
	if (!FJsonObjectConverter::UStructToJsonObjectString(Player, Json))
	{
		ErrorDelegate.ExecuteIfBound({ TEXT("Failed to serializing request") });
		return;
	}

	// TODO(dom) - look at JSON encoding in UE4.
	Json = Json.Replace(TEXT("playerId"), TEXT("playerID"));

	FHttpRequestRef Request = BuildMultiverseRequest("alpha/player/connect", FMultiverseHttpVerb::Post, Json);
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, const FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			TSharedPtr<FJsonObject> JsonObject;

			if (!IsValidJsonResponse(JsonObject, bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound(FMultiverseConnectedResponse(JsonObject));
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::PlayerDisconnect(
	const FString PlayerId, const FPlayerDisconnectDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	const FMultiverseSDKPlayer Player = { PlayerId };
	FString Json;
	if (!FJsonObjectConverter::UStructToJsonObjectString(Player, Json))
	{
		ErrorDelegate.ExecuteIfBound({ TEXT("Failed to serializing request") });
		return;
	}

	// TODO(dom) - look at JSON encoding in UE4.
	Json = Json.Replace(TEXT("playerId"), TEXT("playerID"));

	FHttpRequestRef Request = BuildMultiverseRequest("alpha/player/disconnect", FMultiverseHttpVerb::Post, Json);
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, const FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			TSharedPtr<FJsonObject> JsonObject;

			if (!IsValidJsonResponse(JsonObject, bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound(FMultiverseDisconnectResponse(JsonObject));
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::SetPlayerCapacity(
	const int64 Count, const FSetPlayerCapacityDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	const FMultiversePlayerCapacity PlayerCapacity = { Count };
	FString Json;
	if (!FJsonObjectConverter::UStructToJsonObjectString(PlayerCapacity, Json))
	{
		ErrorDelegate.ExecuteIfBound({ TEXT("Failed to serializing request") });
		return;
	}

	FHttpRequestRef Request = BuildMultiverseRequest("alpha/player/capacity", FMultiverseHttpVerb::Post, Json);
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			if (!IsValidResponse(bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound({});
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::GetPlayerCapacity(FGetPlayerCapacityDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate)
{
	FHttpRequestRef Request = BuildMultiverseRequest("alpha/player/capacity", FMultiverseHttpVerb::Get, "");
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, const FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			TSharedPtr<FJsonObject> JsonObject;

			if (!IsValidJsonResponse(JsonObject, bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound(FMultiverseCountResponse(JsonObject));
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::GetPlayerCount(FGetPlayerCountDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate)
{
	FHttpRequestRef Request = BuildMultiverseRequest("alpha/player/count", FMultiverseHttpVerb::Get, "");
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, const FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			TSharedPtr<FJsonObject> JsonObject;

			if (!IsValidJsonResponse(JsonObject, bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound(FMultiverseCountResponse(JsonObject));
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::IsPlayerConnected(
	const FString PlayerId, const FIsPlayerConnectedDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	FHttpRequestRef Request = BuildMultiverseRequest(
		FString::Format(TEXT("alpha/player/connected/{0}"),
			static_cast<FStringFormatOrderedArguments>(
				TArray<FStringFormatArg, TFixedAllocator<1>>{
		FStringFormatArg(PlayerId)
	}
	)
		),
		FMultiverseHttpVerb::Get,
		""
		);

	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, const FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			TSharedPtr<FJsonObject> JsonObject;

			if (!IsValidJsonResponse(JsonObject, bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound(FMultiverseConnectedResponse(JsonObject));
		});
	Request->ProcessRequest();
}

void UMultiverseSDKComponent::GetConnectedPlayers(
	const FGetConnectedPlayersDelegate SuccessDelegate, const FMultiverseErrorDelegate ErrorDelegate)
{
	FHttpRequestRef Request = BuildMultiverseRequest("alpha/player/connected/{0}", FMultiverseHttpVerb::Get, "");
	Request->OnProcessRequestComplete().BindWeakLambda(this,
		[SuccessDelegate, ErrorDelegate](FHttpRequestPtr HttpRequest, const FHttpResponsePtr HttpResponse, const bool bSucceeded) {
			TSharedPtr<FJsonObject> JsonObject;

			if (!IsValidJsonResponse(JsonObject, bSucceeded, HttpResponse, ErrorDelegate))
			{
				return;
			}

			SuccessDelegate.ExecuteIfBound(FMultiverseConnectedPlayersResponse(JsonObject));
		});
	Request->ProcessRequest();
}

bool UMultiverseSDKComponent::IsValidResponse(const bool bSucceeded, const FHttpResponsePtr HttpResponse, FMultiverseErrorDelegate ErrorDelegate)
{
	if (!bSucceeded)
	{
		ErrorDelegate.ExecuteIfBound({ "Unsuccessful Call" });
		return false;
	}

	if (!EHttpResponseCodes::IsOk(HttpResponse->GetResponseCode()))
	{
		ErrorDelegate.ExecuteIfBound(
			{ FString::Format(TEXT("Error Code - {0}"),
				static_cast<FStringFormatOrderedArguments>(
					TArray<FStringFormatArg, TFixedAllocator<1>>{
						FStringFormatArg(FString::FromInt(HttpResponse->GetResponseCode()))
					})
				)
			}
		);
		return false;
	}

	return true;
}

bool UMultiverseSDKComponent::IsValidJsonResponse(TSharedPtr<FJsonObject>& JsonObject, const bool bSucceeded, const FHttpResponsePtr HttpResponse, FMultiverseErrorDelegate ErrorDelegate)
{
	if (!IsValidResponse(bSucceeded, HttpResponse, ErrorDelegate))
	{
		return false;
	}

	TSharedPtr<FJsonObject> OutObject;
	const FString Json = HttpResponse->GetContentAsString();
	const TSharedRef<TJsonReader<>> JsonReader = TJsonReaderFactory<>::Create(Json);
	if (!FJsonSerializer::Deserialize(JsonReader, OutObject) || !OutObject.IsValid())
	{
		ErrorDelegate.ExecuteIfBound({ FString::Format(TEXT("Failed to parse response - {0}"),
			static_cast<FStringFormatOrderedArguments>(
				TArray<FStringFormatArg, TFixedAllocator<1>>{
					FStringFormatArg(Json)
				})
			)
			});
		return false;
	}

	JsonObject = OutObject.ToSharedRef();
	return true;
}

