// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "Components/ActorComponent.h"
#include "MultiverseSDKClasses.h"
#include "Interfaces/IHttpRequest.h"
#include "WebSockets/Public/IWebSocket.h"

#include "MultiverseSDKComponent.generated.h"

DECLARE_DYNAMIC_DELEGATE_OneParam(FMultiverseErrorDelegate, const FMultiverseSDKError&, Error);

DECLARE_DYNAMIC_DELEGATE_OneParam(FAllocateDelegate, const FMultiverseEmptyResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FGameServerDelegate, const FMultiverseGameServerResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FGetLabelsDelegate, const FGetLabelsResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FGetLabelEnvsDelegate, const FGetLabelEnvsResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FGetConnectedPlayersDelegate, const FMultiverseConnectedPlayersResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FGetPlayerCapacityDelegate, const FMultiverseCountResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FGetPlayerCountDelegate, const FMultiverseCountResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FHealthDelegate, const FMultiverseEmptyResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FIsPlayerConnectedDelegate, const FMultiverseConnectedResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FPlayerConnectDelegate, const FMultiverseConnectedResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FPlayerDisconnectDelegate, const FMultiverseDisconnectResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FReadyDelegate, const FMultiverseEmptyResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FReserveDelegate, const FMultiverseEmptyResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FSetAnnotationDelegate, const FMultiverseEmptyResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FSetLabelDelegate, const FMultiverseEmptyResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FSetPlayerCapacityDelegate, const FMultiverseEmptyResponse&, Response);

DECLARE_DYNAMIC_DELEGATE_OneParam(FShutdownDelegate, const FMultiverseEmptyResponse&, Response);

DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FConnectedDelegate, const FMultiverseGameServerResponse&, Response);


class FMultiverseHttpVerb
{
public:
	enum EVerb
	{
		Get,
		Post,
		Put
	};

	// ReSharper disable once CppNonExplicitConvertingConstructor
	FMultiverseHttpVerb(const EVerb Verb) : Verb(Verb)
	{
	}

	FString ToString() const
	{
		switch (Verb)
		{
		case Post:
			return TEXT("POST");
		case Put:
			return TEXT("PUT");
		case Get:
		default:
			return TEXT("GET");
		}
	}

private:
	const EVerb Verb;
};


UCLASS( ClassGroup=(Custom), meta=(BlueprintSpawnableComponent) )
class MULTIVERSESDK_API UMultiverseSDKComponent : public UActorComponent
{
	GENERATED_BODY()

public:	
	// Sets default values for this component's properties
	UMultiverseSDKComponent();

	UPROPERTY(EditAnywhere, Category = MultiverseSDK, Config)
	FString HttpPort = "9358";
	/**
 * \brief HealthRateSeconds is the frequency to send Health calls. Value of 0 will disable auto health calls.
 */
	UPROPERTY(EditAnywhere, Category = MultiverseSDK, Config)
	float HealthRateSeconds = 10.f;

	/**
 * \brief HealthPing loops calling the Health endpoint.
 * \param RateSeconds rate at which the Health endpoint should be called.
 */
	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Utility")
	void HealthPing(float RateSeconds);

	UPROPERTY(EditAnywhere, Category = MultiverseSDK, Config)
	bool bDisableAutoConnect;

	UPROPERTY(BlueprintAssignable, Category = MultiverseSDK)
	FConnectedDelegate ConnectedDelegate;

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Utility")
	void Connect();

	virtual void EndPlay(const EEndPlayReason::Type EndPlayReason) override;

	virtual void BeginPlay() override;
	/**
 * \brief Allocate self marks this gameserver as Allocated.
 * \param SuccessDelegate - Called on Successful call.
 * \param ErrorDelegate - Called on Unsuccessful call.
 */
	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Lifecycle")
	void Allocate(FAllocateDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Configuration")
	void GameServer(FGameServerDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Configuration")
	void GetLabels(FGetLabelsDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Configuration")
	void GetLabelEnvs(FGetLabelEnvsDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Configuration")
	void WatchGameServer(FGameServerDelegate WatchDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Lifecycle")
	void Health(FHealthDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Lifecycle")
	void Ready(FReadyDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Lifecycle")
	void Reserve(int64 Seconds, FReserveDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Metadata")
	void SetAnnotation(const FString& Key, const FString& Value, FSetAnnotationDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Metadata")
	void SetLabel(const FString& Key, const FString& Value, FSetLabelDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Lifecycle")
	void Shutdown(FShutdownDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Alpha | Player Tracking")
	void GetConnectedPlayers(FGetConnectedPlayersDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Alpha | Player Tracking")
	void GetPlayerCapacity(FGetPlayerCapacityDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Alpha | Player Tracking")
	void GetPlayerCount(FGetPlayerCountDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Alpha | Player Tracking")
	void IsPlayerConnected(FString PlayerId, FIsPlayerConnectedDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Alpha | Player Tracking")
	void PlayerConnect(FString PlayerId, FPlayerConnectDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Alpha | Player Tracking")
	void PlayerDisconnect(FString PlayerId, FPlayerDisconnectDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

	UFUNCTION(BlueprintCallable, Category = "MultiverseSDK | Alpha | Player Tracking")
	void SetPlayerCapacity(int64 Count, FSetPlayerCapacityDelegate SuccessDelegate, FMultiverseErrorDelegate ErrorDelegate);

protected:
	// Called when the game starts
//	virtual void BeginPlay() override;

public:	
	// Called every frame
	virtual void TickComponent(float DeltaTime, ELevelTick TickType, FActorComponentTickFunction* ThisTickFunction) override;

private:
	FTimerHandle HealthTimerHandler;

	FTimerHandle ConnectDelTimerHandle;

	FTimerHandle EnsureWebSocketTimerHandler;

	TSharedPtr<IWebSocket> WatchWebSocket;

	FString WatchMessageBuffer;

	TArray<FGameServerDelegate> WatchGameServerCallbacks;

	FHttpRequestRef BuildMultiverseRequest(
		FString Path = "", const FMultiverseHttpVerb Verb = FMultiverseHttpVerb::Post, const FString Content = "{}");


	static bool IsValidResponse(const bool bSucceeded, const FHttpResponsePtr HttpResponse, FMultiverseErrorDelegate ErrorDelegate);

	static bool IsValidJsonResponse(TSharedPtr<FJsonObject>& JsonObject, const bool bSucceeded, const FHttpResponsePtr HttpResponse, FMultiverseErrorDelegate ErrorDelegate);

	void EnsureWebSocketConnection();

	void HandleWatchMessage(const void* Data, SIZE_T Size, SIZE_T BytesRemaining);

	void DeserializeAndBroadcastWatch(FString const& JsonString);

	UFUNCTION(BlueprintInternalUseOnly)
	void ConnectSuccess(FMultiverseGameServerResponse GameServerResponse);
};
