"use strict";
const { ObjectId } = require("mongodb");
const { newDrawPool, calcLuckLevel } = require("../utils/utils");

// TL;DR game logic
async function queryInfo(users, query) {
  const user = await users.findOne(query, {
    projection: {
      diamonds: 1,
      coins: 1,
      luckPoints: 1,
      drawCounts: 1,
    },
  });
  return {
    diamonds: user.diamonds,
    coins: user.coins,
    luckLevel: calcLuckLevel(user.luckPoints, user.drawCounts),
  };
}

// TL;DR game logic
async function addDiamonds(users, query, diff) {
  const result = await users.findOneAndUpdate(
    query,
    { $inc: { diamonds: diff } },
    {
      returnDocument: "after",
      projection: { diamonds: 1, coins: 1 },
    }
  );
  if (!result) {
    throw new Error("fail to add diamonds");
  }
  return {
    diamonds: result.diamonds,
    coins: result.coins,
  };
}

// TL;DR game logic
async function addCoins(users, query, diff) {
  const result = await users.findOneAndUpdate(
    query,
    { $inc: { coins: diff } },
    {
      returnDocument: "after",
      projection: { diamonds: 1, coins: 1 },
    }
  );
  if (!result) {
    throw new Error("fail to add coins");
  }
  return {
    diamonds: result.diamonds,
    coins: result.coins,
  };
}

// TL;DR game logic
async function getHeros(users, query) {
  const user = await users.findOne(query, {
    projection: { heros: 1 },
  });
  return { type: 0, items: Object.values(user.heros) };
}

// TL;DR game logic
async function getProps(users, query) {
  const user = await users.findOne(query, {
    projection: { props: 1 },
  });
  return { type: 1, items: Object.values(user.props) };
}

// TL;DR game logic
async function drawCards(users, query, count) {
  const user = await users.findOne(query);

  // check enough diamonds?
  if (count > user.diamonds) {
    return { msg: "钻石不够抽卡" };
  }

  // start draw
  let res = [];

  let pool = user.drawPool;
  if (pool.length === 0) {
    pool = newDrawPool();
  }

  if (pool.length < count) {
    const n = pool.length;
    const needN = count - n;
    pool.forEach((e) => {
      res.push(e);
    });
    pool = newDrawPool();
    const lastN = pool.splice(pool.length - needN, needN);
    res = res.concat(lastN);
  } else {
    const lastCount = pool.splice(pool.length - count, count);
    res = res.concat(lastCount);
    res.some((e) => {
      if (e.type === 0) {
        pool = newDrawPool();
        return true;
      }
      return false;
    });
  }

  const { heros, props } = user;
  let coins = 0;
  let diamonds = 0;
  let luckPoints = 0;

  res.forEach((e) => {
    switch (e.type) {
      case 0:
        if (heros[e.name] === undefined) heros[e.name] = e;
        else {
          heros[e.name].count += e.count;
        }
        luckPoints += 100;
        break;
      case 1:
        if (props[e.name] === undefined) props[e.name] = e;
        else {
          props[e.name].count += e.count;
        }
        luckPoints += 1;
        break;
      case 2:
        if (e.name === "diamonds") {
          diamonds += e.count;
          luckPoints += e.count;
        } else if (e.name === "coins") {
          coins += e.count;
          luckPoints += Math.floor((e.count + 3000) / 10000);
        }
        break;
      default:
        // nothing
        break;
    }
  });

  const result = await users.updateOne(query, {
    $inc: {
      diamonds: -count + diamonds,
      coins: coins,
      luckPoints: luckPoints,
      drawCounts: count,
    },
    $set: {
      drawPool: pool,
      heros: heros,
      props: props,
    },
  });

  if (result.matchedCount !== 1) {
    throw new Error("fail to update info after draws");
  }
  return { items: res };
}

exports.main = async (event, context) => {
  // parse body -> (id, type)
  const body = JSON.parse(event.body);
  const { id, action, diff, count } = body;

  console.log("id: ", id, " ", typeof id);
  console.log("action: ", action, " ", typeof action);
  console.log("diff: ", diff, " ", typeof diff);
  console.log("count: ", count, " ", typeof count);

  // db connection
  context.database = require("func-stateless-mongodb-sdk-nodejs").database;

  try {
    // get db connection
    const database = await context.database();
    const client = await database.connection();

    const db = client.db("stateless-demo");
    const users = db.collection("users");

    const query = { _id: new ObjectId(id) };

    // assume user info is created
    // let user;
    // let result;
    // let pool;
    // let res = [];

    switch (action) {
      case 0:
        // query diamonds/coins/luckLevel
        return await queryInfo(users, query);
      case 1:
        // add diamonds
        return await addDiamonds(users, query, diff);
      case 2:
        // add coins
        return await addCoins(users, query, diff);
      case 3:
        // get heros
        return await getHeros(users, query);
      case 4:
        // get props
        return await getProps(users, query);
      case 5:
        // draw cards
        return await drawCards(users, query, count);
      default:
        throw new Error("unknown action to call");
    }
  } catch (e) {
    return { msg: e.message };
  }
};
